{ Install utility for MS Excel Add-Ins }
{ Copyright (c) Edgar Goettel 2008 - 2013 }
{ - install routines - }
unit InstFunc;

interface

uses
  Classes;

procedure GetExcelVersions(aList: TStrings);
function CheckCOMServer(const Server: String): Boolean;
function Install(Remove: Boolean; const AddIn,
  Version: String; Auto: Boolean = False): Integer;

implementation

uses
  Windows, Registry, SysUtils, ActiveX;

{ Detects and provides a List of installed versions }
{ with the "FirstRun" Registry Value                }
procedure GetExcelVersions(aList: TStrings);
var
  r: TRegistry;
  i, j: Integer;
  s: String;
  d: Double;
begin
  r:= TRegistry.Create;
  s:= 'Software\Microsoft\Office';
  if r.OpenKey(s, False) then begin
    r.GetKeyNames(aList);
    r.CloseKey;
    i:= 0;
    while i < aList.Count do begin
      Val(aList[i], d, j);               // check for numeric key names
      if j = 0 then begin
        s:= 'Software\Microsoft\Office\' + aList[i] + '\Excel\Options';
        if r.OpenKey(s, False) then begin
          if r.ValueExists('FirstRun') then    // check for a first run
            inc(i)
          else
            aList.Delete(i);
          r.CloseKey;
        end else
          aList.Delete(i);
      end else
        aList.Delete(i);
    end;
  end;
  r.Free;
end;

function CheckCOMServer(const Server: String): Boolean;
var
  c: TCLSID;
begin
  Result:= CLSIDFromProgID(PWideChar(WideString(Server)), c) = S_OK;
end;

function Openx(const i: Integer): String;
begin
  if i = 0 then
    result:='OPEN'
  else
    result:='OPEN' + IntToStr(i);
end;

{ Compares an Add-In name with a Registry-Entry of an }
{ installed or displayed Add-In                       }
function CmpRegEntry(RegEntry, AddIn: String): Boolean;
var
  s: String;
begin
  Result:= False;
  s:= RegEntry;
  if (s = '') then Exit;
  // strip params if exist
  while (s[1] = '/') and (pos(' ', s) > 0) do
    s:= Copy(s, pos(' ', s) + 1, Length(s));
  if (s = '') then Exit;
  // strip quotes if exist
  if s[1] = '"' then
    s:= Copy(s, 2, Length(s)-2);
  Result:= AnsiSameText(ExtractFileName(s), AddIn);
end;

{ Function Install adds or Removes a given AddIn as }
{ an Add-In of a given MS Excel Version.            }
{ for an Automation Add-In the ProgId of the class  }
{ is passed as AddIn and param Auto is set to True. }
{ Results:                                          }
{ -1  invalid Add-In parameter (empty param or      }
{       invalid file extension)                     }
{ 0   Add-In successful added or removed            }
{ 1   Add-In not found (to remove)                  }
{ 2   failed to delete from Registry                }
{ 3   failed to (over)write Registry                }
{ 4   failed to open Registry                       }
function Install(Remove: Boolean; const AddIn,
  Version: String; Auto: Boolean = False): Integer;
var
  i: Integer;
  r: TRegistry;
  s, lAddIn, AParam: String;
  l: TStrings;
begin
  Result:= -1;      // invalid Add-In param
  lAddIn:=ExtractFileName(AddIn); // search file without path
  if Auto then
    AParam:='/A '   // param for Automation Add-Ins
  else begin
    s:= LowerCase(ExtractFileExt(lAddIn));
    if (s = '.xll') then
      AParam:='/R ' // param for XLL Add-Ins
    else
      AParam:='';
  end;
  if (lAddIn <> '') and ((s = '.xla') or
     (s = '.xlam') or (s = '.xll') or Auto) then begin
    Result:= 1;     // valid params but nothing done
    r:=TRegistry.Create;
    // registry path for active Add-Ins
    s:='Software\Microsoft\Office\' + Version + '\Excel\Options';
    if r.OpenKey(s, False) then begin
      try
        // iterate active add-ins
        // delete matching entry to remove Add-In
        // overwrite matching entry to add/update Add-In
        i:=0;
        while r.ValueExists(Openx(i)) do begin
          if CmpRegEntry(r.ReadString(Openx(i)), lAddIn) then begin
            if Remove then begin // entry found to remove
              if r.DeleteValue(Openx(i)) then begin
                Result:= 0; // successful deleted from Registry
                // rename entries that follow
                while r.ValueExists(Openx(i + 1)) do begin
                  r.RenameValue(Openx(i + 1), Openx(i));
                  inc(i);
                end;
              end else
                Result:= 2; // failed to delete from Registry
            end
            else begin     // entry found to overwrite
              r.WriteString(Openx(i),AParam + '"' + AddIn + '"');
              Result:= 0;  // successful updated Add-In
              Break;       // there's only one matching entry
            end;
          end;
          inc(i);
        end;
      except
        Result:= 3;        // failed to overwrite Registry
      end;
      r.CloseKey;
    end else
      Result:= 4; // failed to open Registry (wrong Excel version?)
    if Result = 1 then begin
      // iterate deactivated add-ins
      // delete from displayed Add-Ins of Add-Manager
      s:='Software\Microsoft\Office\' + Version + '\Excel\Add-in Manager';
      if r.OpenKey(s, False) then begin
        l:= TStringlist.Create;
        r.GetValueNames(l);
        for i:=0 to l.Count-1 do begin
          if CmpRegEntry(l[i], lAddIn) then begin
            if r.DeleteValue(l[i]) then
              Result:= 0  // successful deleted from Registry
            else
              Result:= 2; // failed to delete from Registry
            break;        // there's only one matching entry
          end;
        end;
        l.Free;
        r.CloseKey;
      end;
      if (not Remove) and (Result < 2) then begin
        s:='Software\Microsoft\Office\' + Version + '\Excel\Options';
        if r.OpenKey(s, False) then begin
          try
            // append as last OPEN-entry
            i:=0;
            while r.ValueExists(Openx(i)) do
              inc(i);
            r.WriteString(Openx(i),AParam + '"' + AddIn + '"');
            Result:= 0; // successful added
          except
            Result:= 3; // failed to write to Registry
          end;
          r.CloseKey;
        end else
          Result:= 4;   // failed to open Registry (wrong Excel version?)
      end;
    end;
    r.Free;
  end;
end;

end.
